﻿using RnD.API;
using RnD.Messages.Client;
using RnD.Messages.Interfaces.Data;
using RnD.Model;
using RnD.Model.EF;
using System.Collections.Generic;
using System.Linq;

namespace CheckInfoFields.Services
{
    /// <summary>
    /// Command service for searching and checking info fields
    /// </summary>
    internal static class CheckInfoFieldsService
    {
        #region Methods
        /// <summary>
        /// Search for empty info fields
        /// </summary>
        /// <param name="infoFields"></param>
        /// <returns>List of empty info fields</returns>
        internal static IEnumerable<IInfoField> FindEmptyInfoFields(in IEnumerable<IInfoField> infoFields)
        {
            var emptyInfoFields = new Stack<IInfoField>();
            foreach (var infoField in infoFields)
            {
                if (string.IsNullOrEmpty(infoField.InfoFieldValue))
                {
                    emptyInfoFields.Push(infoField);
                }
            }
            return emptyInfoFields;
        }
        /// <summary>
        /// Sending a message to a user
        /// </summary>
        /// <param name="userId"></param>
        /// <param name="notificationMessage"></param>
        /// <param name="api"></param>
        internal static void SendMessage(in decimal userId, in string notificationMessage, in IAPI api)
        {
            var notificationClient = new NotificationClient();
            var userName = api.User.GetUser(userId).UserName;
            var notificationType = NotificationType.ErrorMessage;
            var connectionName = api.CustomizationSession.ConnectionName;
            var browserGuid = api.CustomizationSession.BrowserGuid;

            MessageService.SendMessage(notificationMessage,
                                       notificationType,
                                       notificationClient,
                                       userName,
                                       connectionName,
                                       browserGuid);
        }
        /// <summary>
        /// Getting all info fields from an entity
        /// </summary>
        /// <param name="entity"></param>
        /// <returns></returns>
        internal static IEnumerable<IInfoField> GetInfoFields(in ICustomizationObject entity)
        {
            var type = entity.GetType();
            var property = type.GetProperty("InfoCards");
            var infoCards = (IReadOnlyCollection<IInfoCard>)property.GetValue(entity, null);
            return infoCards.SelectMany(c => c.InfoFields)
                            .Where(c => c.IsMandatory == RnD.Common.Enums.ThreeState.Yes);
        }
        /// <summary>
        /// Entity status change
        /// </summary>
        /// <param name="entity"></param>
        /// <param name="statusId"></param>
        /// <param name="databaseContext"></param>
        internal static void ChangeStatus(in ICustomizationObject entity, in decimal statusId, in RnDConnection databaseContext)
        {
            dynamic rndvObject;
            if (entity is ISpecification specification)
            {
                rndvObject = databaseContext.RndvSp.Find(new object[] { specification.ID, specification.Version });
            }
            else if (entity is ISample sample)
            {
                rndvObject = databaseContext.RndvSc.Find(new object[] { sample.ID });
            }
            else if (entity is IRequest request)
            {
                rndvObject = databaseContext.RndvRq.Find(new object[] { request.ID });
            }
            else
            {
                return;
            }
            rndvObject.SS = statusId;
            databaseContext.SaveChanges();
        }
        /// <summary>
        /// Getting the display title from the UI
        /// </summary>
        /// <param name="entity"></param>
        /// <param name="infoFields"></param>
        /// <param name="databaseContext"></param>
        /// <returns></returns>
        internal static IEnumerable<string> GetInfoFieldDisplayTitles(in ICustomizationObject entity, in IEnumerable<IInfoField> infoFields, in RnDConnection databaseContext)
        {
            var names = new List<string>();
            foreach (var infoField in infoFields)
            {
                string name = null;
                if (entity is ISpecification specification)
                {
                    name = databaseContext.RndvSpIiLang.FirstOrDefault(c => c.SP == specification.ID &&
                                                                            c.SP_VERSION == specification.Version &&
                                                                            c.IINODE == infoField.InfoFieldNodeNumber &&
                                                                            c.ICNODE == infoField.InfoCardNodeNumber)?.DSP_TITLE;
                }
                else if (entity is ISample sample)
                {
                    name = databaseContext.RndvScIiLang.FirstOrDefault(c => c.SC == sample.ID && 
                                                                            c.II == infoField.ID &&
                                                                            c.ICNODE == infoField.InfoCardNodeNumber &&
                                                                            c.IINODE == infoField.InfoFieldNodeNumber)?.DSP_TITLE;
                }
                else if (entity is IRequest request)
                {
                    name = databaseContext.RndvRqIiLang.FirstOrDefault(c => c.RQ == request.ID && 
                                                                            c.II == infoField.ID &&
                                                                            c.ICNODE == infoField.InfoCardNodeNumber &&
                                                                            c.IINODE == infoField.InfoFieldNodeNumber)?.DSP_TITLE;
                }

                names.Add(name ?? infoField.DisplayTitle);
            }

            return names;
        }
        #endregion
    }
}
