﻿using IntegrationFTS.Services;
using NLog;
using RnD.API;
using RnD.Common.Enums;
using RnD.Common.Model.ActionBarButtons;
using RnD.Messages.Client;
using RnD.Messages.Interfaces.Data;
using RnD.Model;
using RnD.Model.EF;
using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using XmlGenerateFTS.Extensions;
using XmlGenerateFTS.Helpers;
using XmlGenerateFTS.Models;
using XmlGenerateFTS.Models.Xml;

namespace XmlGenerateFTS.Services
{
    /// <summary>
    /// List of commands for generate FTS xml 
    /// </summary>
    internal class FTSService
    {
        #region Variables 
        private const int _languageId = 2;
        private static readonly Logger _logger = LogManager.GetCurrentClassLogger();
        #endregion

        #region Methods
        public static void ExportFTSAction(ICustomizationObject aCustomisationObject, IAPI api, RnDConnection databaseContext)
        {
            try
            {
                if (!(aCustomisationObject is ISample sampleEntity))
                {
                    _logger.Info("Object is not a sample");
                    return;
                }
                var sampleId = sampleEntity.ID;
                ISample sample = api.Sample.GetSample(sampleId);
                var result = ExportFTS(sample, databaseContext);

                var notificationClient = new NotificationClient();
                var notificationType = result.MessageType == MessageType.Information ? NotificationType.Message : NotificationType.ErrorMessage;

                MessageService.SendMessage(result.Message,
                                           notificationType,
                                           notificationClient,
                                           api.CustomizationSession.User.UserName,
                                           api.CustomizationSession.ConnectionName,
                                           api.CustomizationSession.BrowserGuid);
                return;
            }
            catch (Exception ex)
            {
                _logger.Error(ex);
                return;
            }
        }
        public static ActionBarButtonResult ExportFTSButton(List<ICustomizationObject> aCustomisationObject, IAPI api, RnDConnection databaseContext)
        {
            try
            {
                if (!(aCustomisationObject.FirstOrDefault(c => c is ISample) is ISample sampleEntity))
                {
                    _logger.Info("Object is not a sample");
                    return new ActionBarButtonResult { Message = $"Объект не является пробой!", MessageType = MessageType.Information };
                }

                var sampleId = sampleEntity.ID;
                ISample sample = api.Sample.GetSample(sampleId);
                var result = ExportFTS(sample, databaseContext);
                return result;
            }
            catch (Exception ex)
            {
                _logger.Error(ex);
                var result = new ActionBarButtonResult { Message = $"Ошибка: {ex.Message}", MessageType = MessageType.Error };
                return result;
            }
        }
        private static ActionBarButtonResult ExportFTS(ISample sample, RnDConnection databaseContext)
        {
            var sampleId = sample.ID;
            var ssQrIssued = databaseContext.RndvSs.Find(9);

            if (sample.Status != ssQrIssued.SS)
            {
                _logger.Info(@"The status ""Passport issued"" is required for export");
                return new ActionBarButtonResult { Message = @"Для формирования ФТС необходим статус : ""Паспорт выписан""", MessageType = MessageType.Error };
            }

            var infoFieldFts = sample.InfoCards
                                  .FirstOrDefault(c => c.ShortDescription == "ICQDataSheet")?
                                  .InfoFields?
                                  .FirstOrDefault(c => c.ShortDescription == "FTS");

            if (infoFieldFts == null || infoFieldFts.InfoFieldValue != "1")
            {
                _logger.Info(@"Not found InfoField FTS");
                return new ActionBarButtonResult { Message = @"Не найдено инфополе ФТС", MessageType = MessageType.Error };
            }

            var sqlQueryLayoutTable = $"SELECT * FROM [RndSuite].[PasportLayoutTable] WHERE [SC] = @SampleId AND [languageId] = @LanguageId";
            var sqlParamLayoutSampleId = new SqlParameter("@SampleId", sampleId);
            var sqlParamLayoutLanguageId = new SqlParameter("@LanguageId", _languageId);
            var pasportLayoutTableModel = databaseContext.Database.SqlQuery<PasportLayoutTableModel>(sqlQueryLayoutTable, sqlParamLayoutSampleId, sqlParamLayoutLanguageId)
                                                                  .FirstOrDefault();

            var sqlQueryMainTable = GetSelectRequest("PasportMainTable", "SC", "@SampleId");
            var sqlParamMainSampleId = new SqlParameter("@SampleId", sampleId);
            var passportMainTableModels = databaseContext.Database.SqlQuery<PasportMainTableModel>(sqlQueryMainTable, sqlParamMainSampleId)
                                                                  .ToList();

            var sqlQuerySecondTable = GetSelectRequest("PasportSecondTable", "SC", "@SampleId");
            var sqlParamSecondSampleId = new SqlParameter("@SampleId", sampleId);
            var pasportSecondTableModels = databaseContext.Database.SqlQuery<PasportSecondTableModel>(sqlQuerySecondTable, sqlParamSecondSampleId)
                                                                   .ToList();

            if (pasportLayoutTableModel == null || passportMainTableModels.Count == 0 || pasportSecondTableModels.Count == 0)
            {
                _logger.Info($"XML file was not generated, missing data for this sample: {sample.Description}");
                return new ActionBarButtonResult { Message = $"XML файл не был сформирован, отсутствуют данные по данной пробе: {sample.Description}", MessageType = MessageType.Information };
            }
            var sqlQueryBranchContact = $"SELECT * FROM [RndSuite].[SiemtRepBranchContact] WHERE [BRANCH_ID] = @LocationId AND [LANG_ID] = @LanguageId";
            var sqlParamBranchLocationId = new SqlParameter("@LocationId", pasportLayoutTableModel.locationId);
            var sqlParamBranchLayoutLanguageId = new SqlParameter("@LanguageId", _languageId);
            var siemtRepBranchContactModel = databaseContext.Database.SqlQuery<SiemtRepBranchContactModel>(sqlQueryBranchContact, sqlParamBranchLocationId, sqlParamBranchLayoutLanguageId)
                                                                     .FirstOrDefault();


            var sqlQuerySavePath = $"SELECT SaveFilePath FROM [RndSuite].[IntegrationFts] WHERE [Id] = @Id";
            var sqlParamSavePathId = new SqlParameter("@Id", 1);
            var saveFolderPath = databaseContext.Database.SqlQuery<string>(sqlQuerySavePath, sqlParamSavePathId)
                                                         .FirstOrDefault();

            if (!Directory.Exists(saveFolderPath)) Directory.CreateDirectory(saveFolderPath);
            string saveFilePath = $@"{saveFolderPath}/QP-{pasportLayoutTableModel.languageName}-{pasportLayoutTableModel.passportNumber}.xml";

            FTSService.GenerateFreeDocXmlFile(pasportLayoutTableModel, passportMainTableModels, pasportSecondTableModels,
                                               siemtRepBranchContactModel, saveFilePath);
            var infoField = sample.InfoCards
                                  .FirstOrDefault(c => c.ShortDescription == "ICQDataSheet")?
                                  .InfoFields?
                                  .FirstOrDefault(c => c.ShortDescription == "SendStatusFTS");
            if (infoField != null) infoField.InfoFieldValue = "Success";

            return new ActionBarButtonResult { Message = $"Успешное формирование XML файла, файл был сохранен по пути: {saveFolderPath}", MessageType = MessageType.Information };

        }

        /// <summary>
        /// Get line of select request from RndSuite
        /// </summary>
        /// <param name="tableName">Name of table</param>
        /// <param name="columnName">Name of column</param>
        /// <param name="paramName">Value for search</param>
        /// <returns>Line select requiest</returns>
        private static string GetSelectRequest(string tableName, string columnName, object paramName) => $"SELECT * FROM [RndSuite].[{tableName}] WHERE [{columnName}] = {paramName}";
        /// <summary>
        /// Create XML file format an FreeDoc
        /// </summary>
        /// <param name="pasportLayoutTableModel">Model pasport layout</param>
        /// <param name="pasportMainTableModels">Models pasport main table</param>
        /// <param name="pasportSecondTableModels">Models pasport second table</param>
        /// <param name="saveFilePath">Full save file path</param>
        /// <returns>Generated XML file format an FreeDoc/returns>
        private static FileInfo GenerateFreeDocXmlFile(PasportLayoutTableModel pasportLayoutTableModel,
                                             List<PasportMainTableModel> pasportMainTableModels,
                                             List<PasportSecondTableModel> pasportSecondTableModels,
                                             SiemtRepBranchContactModel siemtRepBranchContactModel,
                                             string saveFilePath)
        {
            //TODO: Custom action
            //SamplePlace_ new parameter on info card - Место отбора проб
            //Adress manufacture - BranchContact
            // product required + RegulatoryDocument + if is not null OKPD2 + and ТР ТС 030/2012

            #region Initialize FreeDoc model

            string documentModelId = "1006088е";
            var linesFulleName = pasportLayoutTableModel.userFullName.Split('.');
            string name = "-", middleName = "-", surname = "-";
            if (linesFulleName.Length == 3)
            {
                name = linesFulleName[0];
                middleName = linesFulleName[1];
                surname = linesFulleName[2];
            }
            FreeDoc freeDocModel = new FreeDoc
            {
                DocumentID = $"{pasportLayoutTableModel.SC}",
                DocumentModelID = documentModelId,
                DocumentBody = new DocumentBody
                {
                    TextSections = new List<TextSection>(),
                    Table = new List<Table>(),
                },
                DocumentHead = new DocumentHead
                {
                    DocumentNumber = pasportLayoutTableModel.passportNumber,
                    DocumentName = "Паспорт качества", //BranchContact <-> LocationId,
                    OrganizationName = "ООО «ЛЛК-Интернешнл»"
                },
                DocumentSign = new DocumentSign
                {
                    PersonSurname = surname,
                    PersonName = name,
                    PersonMiddleName = middleName,
                    PersonPost = pasportLayoutTableModel.userPosition,
                },
            };
            var documentDate = pasportLayoutTableModel.passportIssueDate;
            if(!string.IsNullOrEmpty(documentDate) && documentDate != "-")
            {
                freeDocModel.DocumentHead.DocumentDate = DateTime.ParseExact(documentDate, "dd.MM.yy", null).ToString("yyyy-MM-dd");
            }
            #endregion

            #region Main Table
            Table tableMainModel = new Table();
            tableMainModel.FillModelMainTable(pasportLayoutTableModel, pasportMainTableModels);
            freeDocModel.DocumentBody.Table.Add(tableMainModel);
            #endregion

            #region Second Table
            Table tableSecondModel = new Table();
            tableSecondModel.FillModelSecondTable(pasportSecondTableModels);
            freeDocModel.DocumentBody.Table.Add(tableSecondModel);
            #endregion

            #region Layout Table
            //Hardcode
            freeDocModel.DocumentBody.TextSections.Add(new TextSection { TextPara = "СМК сертифицирована: ISO 9001" });
            var exeptionNameProperties = new List<string> { "SC", "userFullName", "userPosition", "passportIssueDate", "passportNumber", "locationId", "languageName" };

            pasportLayoutTableModel?.GetType()
                                    .GetProperties()
                                    .Where(property => !exeptionNameProperties.Contains(property.Name) && !property.GetValue(pasportLayoutTableModel).ToString().IsNullOrEmpty())
                                    .ToList()
                                    .ForEach(property => freeDocModel.DocumentBody.TextSections.Add(
                                     new TextSection
                                     {
                                         SectionName = property.CustomAttributes.FirstOrDefault().ConstructorArguments.FirstOrDefault().Value.ToString(),
                                         TextPara = property.GetValue(pasportLayoutTableModel).ToString().ReplaceRequiredValues(),
                                     }));


            //if(!selectionPlace.IsNullOrEmpty())
            //{
            //    freeDocModel.DocumentBody.TextSections.Add(
            //    new TextSection
            //    {
            //        TextPara = selectionPlace
            //    });
            //}
            if(siemtRepBranchContactModel != null)
            {
                freeDocModel.DocumentBody.TextSections.Add(
                new TextSection
                {
                    TextPara = siemtRepBranchContactModel.MANUFACTURE_ADDRESS
                });
                freeDocModel.DocumentBody.TextSections.Add(
                new TextSection
                {
                    TextPara = siemtRepBranchContactModel.LAB_NAME
                });
            }
            if (pasportLayoutTableModel.declarationEAEU.IsNullOrEmpty()) freeDocModel.DocumentBody.TextSections.Add(
            new TextSection
            {
                TextPara = $"Продукт соответствует требованиям: {pasportLayoutTableModel.regulatoryDocument}."
            });
            else freeDocModel.DocumentBody.TextSections.Add(
            new TextSection
            {
                TextPara = $"Продукт соответствует требованиям: {pasportLayoutTableModel.regulatoryDocument} и ТР ТС 030/2012."
            });



            #endregion

            return Serialize(freeDocModel, saveFilePath);
        }
        /// <summary>
        /// Serialize FreeDocModel to Xml
        /// </summary>
        /// <param name="freeDocModel"></param>
        /// <param name="saveFilePath"></param>
        /// <returns></returns>
        private static FileInfo Serialize(FreeDoc freeDocModel, string saveFilePath)
        {
            XmlSerializer xsav = new XmlSerializer(typeof(FreeDoc));
            XmlSerializerNamespaces xmlSerializerNamespaces = new XmlSerializerNamespaces();
            xmlSerializerNamespaces.Add("cat_ru", FreeDocNamespace.CommonAggregateTypesNamespace);
            xmlSerializerNamespaces.Add("clt_ru", FreeDocNamespace.CommonLeafTypesNamespace);
            xmlSerializerNamespaces.Add("CategoryCust", FreeDocNamespace.CategoriesNamespace);
            
            using (StreamWriter sw = new StreamWriter(saveFilePath, false, Encoding.UTF8))
            {
                xsav.Serialize(sw, freeDocModel, xmlSerializerNamespaces);
            }

            return new FileInfo(saveFilePath);
        }
        #endregion
    }
}
