﻿using IntegrationKSSS.WebService.Infrastructure.Cryptography;
using IntegrationKSSS.WebService.Infrastructure.Data;
using IntegrationKSSS.WebService.Infrastructure.Handlers;
using IntegrationKSSS.WebService.Infrastructure.Repositories;
using IntegrationKSSS.WebService.Infrastructure.Services;
using Microsoft.AspNetCore.Authentication;
using Microsoft.EntityFrameworkCore;
using NLog;
using NLog.Web;
using RouteHandler = IntegrationKSSS.WebService.Infrastructure.RouteHandlers.RouteHandler;
using IRouteHandler = IntegrationKSSS.WebService.Infrastructure.RouteHandlers.IRouteHandler;
using NLog.Config;
using ILogger = NLog.ILogger;
using System.Reflection;

namespace IntegrationKSSS.WebService.Infrastructure.Configurations
{
    public static class ApplicationConfiguration
    {
        public static ILogger GetLogger()
        {
            string fileName = string.Empty;
#if DEBUG
            fileName = Path.GetFullPath(@"../../IntegrationKSSS/IntegrationKSSS.WebService/NLog.config");
#endif
#if !DEBUG
            fileName = $"{Path.GetDirectoryName(Assembly.GetEntryAssembly().Location)}/NLog.config";
#endif


            LogManager.Configuration = new XmlLoggingConfiguration(fileName);
            return LogManager.GetCurrentClassLogger();
        }

        public static void RegisterBuilder(WebApplicationBuilder webApplicationBuilder)
        {
            var logger = GetLogger();

            try
            {
                webApplicationBuilder.Services.AddEndpointsApiExplorer();
                webApplicationBuilder.Logging.ClearProviders();
                webApplicationBuilder.Host.UseNLog();
                webApplicationBuilder.Services.AddSwaggerGen();
                webApplicationBuilder.Services.AddSingleton(logger);
                webApplicationBuilder.Services.AddAuthorization();
                webApplicationBuilder.Services.AddAuthentication("BasicAuthentication")
                                              .AddScheme<AuthenticationSchemeOptions, BasicAuthenticationHandler>("BasicAuthentication", null);

                var connectionString = webApplicationBuilder.Configuration["ConnectionStrings:MicrosoftSqlServer"]
                                                            .DecryptString(AesConfiguration.Key, AesConfiguration.IV);
                webApplicationBuilder.Services.AddDbContext<RDnLDbContext>(options =>
                                                                           options.UseSqlServer(connectionString));
                webApplicationBuilder.Services.AddScoped<IAuthService, AuthService>();
                webApplicationBuilder.Services.AddScoped<IKsssRepository, KsssRepository>();
                webApplicationBuilder.Services.AddTransient<IRouteHandler, RouteHandler>();
            }
            catch (Exception ex)
            {
                logger.Error(ex, "Stopped web api because of exception");
                throw;
            }
            finally
            {
                LogManager.Shutdown();
            }

        }
        public static void RegisterApplication(WebApplication webApplication)
        {
            if (webApplication.Environment.IsDevelopment())
            {
                webApplication.UseSwagger();
                webApplication.UseSwaggerUI();
            }
            webApplication.UseAuthorization();
            webApplication.UseAuthentication();
            webApplication.Services.GetServices<IRouteHandler>().ToList().ForEach(c => c.Register(webApplication));

        }
    }
}
